/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.service.http.commons;

import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.service.http.commons.ResourceLoader;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.client.http.DataLoader;
import eu.europa.esig.dss.spi.client.http.Protocol;
import eu.europa.esig.dss.utils.Utils;
import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileCacheDataLoader
implements DataLoader {
    private static final Logger LOG = LoggerFactory.getLogger(FileCacheDataLoader.class);
    private File fileCacheDirectory = new File(System.getProperty("java.io.tmpdir"));
    private ResourceLoader resourceLoader = new ResourceLoader();
    private List<String> toBeLoaded;
    private List<String> toIgnored;
    private Long cacheExpirationTime;
    private DataLoader dataLoader;

    public FileCacheDataLoader() {
    }

    public FileCacheDataLoader(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public DataLoader getDataLoader() {
        return this.dataLoader;
    }

    public void setDataLoader(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public void setFileCacheDirectory(File fileCacheDirectory) {
        this.fileCacheDirectory = fileCacheDirectory;
        this.fileCacheDirectory.mkdirs();
    }

    public void setCacheExpirationTime(long cacheExpirationTimeInMilliseconds) {
        this.cacheExpirationTime = cacheExpirationTimeInMilliseconds;
    }

    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
    }

    public void addToBeLoaded(String url) {
        if (this.toBeLoaded == null) {
            this.toBeLoaded = new ArrayList<String>();
        }
        if (Utils.isStringNotBlank(url)) {
            this.toBeLoaded.add(url);
        }
    }

    public void addToBeIgnored(String urlString) {
        if (this.toIgnored == null) {
            this.toIgnored = new ArrayList<String>();
        }
        if (Utils.isStringNotBlank(urlString)) {
            String normalizedFileName = ResourceLoader.getNormalizedFileName(urlString);
            this.toIgnored.add(normalizedFileName);
        }
    }

    @Override
    public byte[] get(String url, boolean refresh) {
        if (this.toBeLoaded != null && !this.toBeLoaded.contains(url)) {
            return null;
        }
        String fileName = ResourceLoader.getNormalizedFileName(url);
        File file = this.getCacheFile(fileName);
        boolean fileExists = file.exists();
        boolean isCacheExpired = this.isCacheExpired(file);
        if (fileExists && !refresh && !isCacheExpired) {
            LOG.debug("Cached file was used");
            byte[] bytes = DSSUtils.toByteArray(file);
            return bytes;
        }
        if (!fileExists) {
            LOG.debug("There is no cached file!");
        } else {
            LOG.debug("The refresh is forced!");
        }
        byte[] bytes = null;
        bytes = !this.isNetworkProtocol(url) ? this.getLocalFileContent(url) : this.dataLoader.get(url);
        if (Utils.isArrayNotEmpty(bytes)) {
            File out = this.getCacheFile(fileName);
            DSSUtils.saveToFile(bytes, out);
        }
        return bytes;
    }

    private byte[] getLocalFileContent(String urlString) {
        byte[] returnedBytes = null;
        String resourcePath = this.resourceLoader.getAbsoluteResourceFolder(urlString.trim());
        if (resourcePath != null) {
            File fileResource = new File(resourcePath);
            returnedBytes = DSSUtils.toByteArray(fileResource);
        }
        return returnedBytes;
    }

    @Override
    public byte[] get(String url) {
        return this.get(url, false);
    }

    protected boolean isNetworkProtocol(String urlString) {
        String normalizedUrl = urlString.trim().toLowerCase();
        return Protocol.isHttpUrl(normalizedUrl) || Protocol.isLdapUrl(normalizedUrl) || Protocol.isFtpUrl(normalizedUrl);
    }

    private File getCacheFile(String fileName) {
        String trimmedFileName = fileName.trim();
        if (this.toIgnored != null && this.toIgnored.contains(trimmedFileName)) {
            throw new DSSException("Part of urls to ignore.");
        }
        LOG.debug("Cached file: {}/{}", (Object)this.fileCacheDirectory, (Object)trimmedFileName);
        File file = new File(this.fileCacheDirectory, trimmedFileName);
        return file;
    }

    public byte[] loadFileFromCache(String urlString) {
        String fileName = ResourceLoader.getNormalizedFileName(urlString);
        File file = this.getCacheFile(fileName);
        if (file.exists()) {
            byte[] bytes = DSSUtils.toByteArray(file);
            return bytes;
        }
        return null;
    }

    public void saveBytesInCache(String urlString, byte[] bytes) {
        String fileName = ResourceLoader.getNormalizedFileName(urlString);
        File out = this.getCacheFile(fileName);
        DSSUtils.saveToFile(bytes, out);
    }

    @Override
    public byte[] post(String urlString, byte[] content) throws DSSException {
        String fileName = ResourceLoader.getNormalizedFileName(urlString);
        byte[] digest = DSSUtils.digest(DigestAlgorithm.MD5, content);
        String digestHexEncoded = DSSUtils.toHex(digest);
        String cacheFileName = fileName + "." + digestHexEncoded;
        File file = this.getCacheFile(cacheFileName);
        boolean fileExists = file.exists();
        boolean isCacheExpired = this.isCacheExpired(file);
        if (fileExists && !isCacheExpired) {
            LOG.debug("Cached file was used");
            byte[] byteArray = DSSUtils.toByteArray(file);
            return byteArray;
        }
        LOG.debug("There is no cached file!");
        byte[] returnedBytes = null;
        if (this.isNetworkProtocol(urlString)) {
            returnedBytes = this.dataLoader.post(urlString, content);
        }
        if (Utils.isArrayNotEmpty(returnedBytes)) {
            File cacheFile = this.getCacheFile(cacheFileName);
            DSSUtils.saveToFile(returnedBytes, cacheFile);
        }
        return returnedBytes;
    }

    private boolean isCacheExpired(File file) {
        if (this.cacheExpirationTime == null) {
            return false;
        }
        if (!file.exists()) {
            return true;
        }
        long currentTime = new Date().getTime();
        if (file.lastModified() + this.cacheExpirationTime < currentTime) {
            LOG.debug("Cache is expired");
            return true;
        }
        return false;
    }

    @Override
    public DataLoader.DataAndUrl get(List<String> urlStrings) {
        int numberOfUrls = urlStrings.size();
        int ii = 0;
        for (String urlString : urlStrings) {
            try {
                ++ii;
                byte[] bytes = this.get(urlString);
                if (bytes == null) continue;
                return new DataLoader.DataAndUrl(bytes, urlString);
            }
            catch (Exception e) {
                if (ii == numberOfUrls) {
                    if (e instanceof DSSException) {
                        throw (DSSException)e;
                    }
                    throw new DSSException(e);
                }
                LOG.warn("Impossible to obtain data using {}", (Object)urlString, (Object)e);
            }
        }
        return null;
    }

    @Override
    public void setContentType(String contentType) {
        this.dataLoader.setContentType(contentType);
    }
}

